document.addEventListener("DOMContentLoaded", () => {
    const crosswordData = {
        words: [
            { id: 1, word: "PROCESSEUR", direction: "horizontal", row: 0, col: 0, clue: "Cœur de l'ordinateur qui exécute les instructions" },
            { id: 2, word: "ORDINATEUR", direction: "vertical", row: 0, col: 2, clue: "Appareil permettant d'effectuer des calculs et de traiter des informations" },
            { id: 3, word: "SOURIS", direction: "horizontal", row: 4, col: 5, clue: "Dispositif de pointage permettant d'interagir avec l'interface graphique d'un ordinateur" },
            { id: 4, word: "SAUVEGARDE", direction: "vertical", row: 2, col: 7, clue: "Action de conserver des données pour éviter leur perte" },
            { id: 5, word: "BIT", direction: "horizontal", row: 3, col: 1, clue: "Unité de base de l'information en informatique, représentant un 0 ou un 1" },
            { id: 6, word: "FIREWALL", direction: "vertical", row: 3, col: 9, clue: "Système de sécurité permettant de filtrer les connexions réseau" },
            { id: 7, word: "PYTHON", direction: "horizontal", row: 6, col: 0, clue: "Langage de programmation très utilisé dans l'informatique et la science des données" },
            { id: 8, word: "WORD", direction: "horizontal", row: 9, col: 0, clue: "Logiciel de traitement de texte, faisant partie de la suite Microsoft Office" },
            { id: 9, word: "CODE", direction: "vertical", row: 5, col: 4, clue: "Ensemble d'instructions écrites dans un langage de programmation pour exécuter des tâches" }
        ]
    };


    const gridContainer = document.getElementById("grille");
    const horizontalCluesDiv = document.getElementById("horizontal-clues");
    const verticalCluesDiv = document.getElementById("vertical-clues");
    const verifierButton = document.getElementById("verifier");
    const reinitialiserButton = document.getElementById("reinitialiser");
    const messageDiv = document.getElementById("message");

    // Calculate grid dimensions
    let maxRow = 0;
    let maxCol = 0;
    crosswordData.words.forEach(word => {
        const wordLength = word.word.length;
        if (word.direction === 'horizontal') {
            maxRow = Math.max(maxRow, word.row);
            maxCol = Math.max(maxCol, word.col + wordLength - 1);
        } else {
            maxRow = Math.max(maxRow, word.row + wordLength - 1);
            maxCol = Math.max(maxCol, word.col);
        }
    });

    // Create optimized grid
    const grid = Array(maxRow + 1).fill(null).map(() => 
        Array(maxCol + 1).fill(null).map(() => ({
            letter: '',
            isActive: false,
            number: null
        }))
    );

    // Mark active cells and add numbers
    crosswordData.words.forEach(word => {
        const { row, col, direction, id, word: letters } = word;
        
        // Add number to first cell
        grid[row][col].number = id;
        
        // Mark cells as active
        letters.split('').forEach((letter, index) => {
            const currentRow = direction === 'horizontal' ? row : row + index;
            const currentCol = direction === 'horizontal' ? col + index : col;
            grid[currentRow][currentCol].isActive = true;
        });
    });

    // Create grid HTML
    const table = document.createElement('table');
    table.className = 'crossword-grid';

    for (let i = 0; i <= maxRow; i++) {
        const row = document.createElement('tr');
        for (let j = 0; j <= maxCol; j++) {
            const cell = document.createElement('td');
            if (grid[i][j].isActive) {
                const input = document.createElement('input');
                input.type = 'text';
                input.maxLength = 1;
                input.dataset.row = i;
                input.dataset.col = j;
                input.className = 'crossword-cell';
                
                if (grid[i][j].number) {
                    const number = document.createElement('span');
                    number.className = 'cell-number';
                    number.textContent = grid[i][j].number;
                    cell.appendChild(number);
                }
                
                cell.appendChild(input);
                cell.className = 'active-cell';
            } else {
                cell.className = 'inactive-cell';
            }
            row.appendChild(cell);
        }
        table.appendChild(row);
    }
    gridContainer.appendChild(table);

    // Create clues
    crosswordData.words.forEach(word => {
        const clueElement = document.createElement('p');
        clueElement.innerHTML = `<strong>${word.id}.</strong> ${word.clue}`;
        if (word.direction === 'horizontal') {
            horizontalCluesDiv.appendChild(clueElement);
        } else {
            verticalCluesDiv.appendChild(clueElement);
        }
    });

    // Add automatic movement to next cell after input
    const inputs = document.querySelectorAll('.crossword-cell');
    inputs.forEach(input => {
        input.addEventListener('input', (e) => {
            if (e.target.value) {
                const row = parseInt(input.dataset.row);
                const col = parseInt(input.dataset.col);
                
                // Try to find next input in current direction
                let nextInput = document.querySelector(`input[data-row="${row}"][data-col="${col + 1}"]`);
                if (!nextInput) {
                    nextInput = document.querySelector(`input[data-row="${row + 1}"][data-col="${col}"]`);
                }
                if (nextInput) {
                    nextInput.focus();
                }
            }
        });
    });

    // Verification logic
    verifierButton.addEventListener('click', () => {
        let isCorrect = true;
        const inputs = document.querySelectorAll('.crossword-cell');
        
        inputs.forEach(input => {
            const row = parseInt(input.dataset.row);
            const col = parseInt(input.dataset.col);
            const value = input.value.toUpperCase();
            
            // Find the word this cell belongs to
            const word = crosswordData.words.find(w => {
                const cells = w.word.split('').map((_, i) => ({
                    row: w.direction === 'horizontal' ? w.row : w.row + i,
                    col: w.direction === 'horizontal' ? w.col + i : w.col
                }));
                return cells.some(cell => cell.row === row && cell.col === col);
            });

            if (word) {
                const index = word.direction === 'horizontal' ? col - word.col : row - word.row;
                const correctLetter = word.word[index];
                
                if (value !== correctLetter) {
                    isCorrect = false;
                    input.classList.remove('correct');
                    input.classList.add('incorrect');
                } else {
                    input.classList.remove('incorrect');
                    input.classList.add('correct');
                }
            }
        });

        messageDiv.textContent = isCorrect ? 'Bravo ! Toutes les réponses sont correctes !' : 'Il y a encore des erreurs. Continuez !';
        messageDiv.className = isCorrect ? 'success' : 'error';
    });

    // Reset logic
    reinitialiserButton.addEventListener('click', () => {
        const inputs = document.querySelectorAll('.crossword-cell');
        inputs.forEach(input => {
            input.value = '';
            input.classList.remove('incorrect');
            input.classList.remove('correct');
        });
        messageDiv.textContent = '';
        messageDiv.className = '';
    });

    // Keyboard navigation
    inputs.forEach(input => {
        input.addEventListener('keyup', (e) => {
            const row = parseInt(input.dataset.row);
            const col = parseInt(input.dataset.col);
            
            if (e.key === 'ArrowRight') {
                const nextInput = document.querySelector(`input[data-row="${row}"][data-col="${col + 1}"]`);
                if (nextInput) nextInput.focus();
            } else if (e.key === 'ArrowLeft') {
                const prevInput = document.querySelector(`input[data-row="${row}"][data-col="${col - 1}"]`);
                if (prevInput) prevInput.focus();
            } else if (e.key === 'ArrowDown') {
                const downInput = document.querySelector(`input[data-row="${row + 1}"][data-col="${col}"]`);
                if (downInput) downInput.focus();
            } else if (e.key === 'ArrowUp') {
                const upInput = document.querySelector(`input[data-row="${row - 1}"][data-col="${col}"]`);
                if (upInput) upInput.focus();
            }
        });
    });
});